<?php

namespace App\Filament\Resources\TopUps;

use App\Filament\Resources\TopUps\Pages\CreateTopUp;
use App\Filament\Resources\TopUps\Pages\EditTopUp;
use App\Filament\Resources\TopUps\Pages\ListTopUps;
use App\Models\TopUpOption;
use BackedEnum;
use Filament\Actions;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use UnitEnum;

class TopUpResource extends Resource
{
    protected static ?string $model = TopUpOption::class;

    protected static BackedEnum|string|null $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationLabel = 'Top Up Options';

    protected static ?string $modelLabel = 'Top Up Option';

    protected static ?string $pluralModelLabel = 'Top Up Options';

    public static function getNavigationGroup(): ?string
    {
        return 'Billing';
    }

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->schema([
                // Basic Information
                \Filament\Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255)
                    ->label('Package Title')
                    ->helperText('Name of the top-up package'),
                
                \Filament\Forms\Components\TextInput::make('words')
                    ->numeric()
                    ->required()
                    ->label('Word Count')
                    ->helperText('Number of words in this package'),
                
                \Filament\Forms\Components\TextInput::make('price')
                    ->numeric()
                    ->required()
                    ->prefix('₹')
                    ->step(0.01)
                    ->label('Package Price')
                    ->helperText('Price for this word package'),
                
                // Description
                \Filament\Forms\Components\Textarea::make('description')
                    ->rows(3)
                    ->label('Description')
                    ->helperText('Package description for customers')
                    ->maxLength(255),
                
                // Status and Flags
                \Filament\Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->helperText('Whether this package is available for purchase'),
                
                \Filament\Forms\Components\Toggle::make('is_best_value')
                    ->label('Best Value')
                    ->default(false)
                    ->helperText('Mark this as the best value package'),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('words')
                    ->label('Words')
                    ->numeric()
                    ->sortable()
                    ->formatStateUsing(fn ($state) => number_format($state)),
                
                Tables\Columns\TextColumn::make('price')
                    ->label('Price')
                    ->money('INR')
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('description')
                    ->label('Description')
                    ->limit(50)
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\IconColumn::make('is_best_value')
                    ->label('Best Value')
                    ->boolean()
                    ->sortable(),
                
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->sortable(),
                
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                
                Tables\Columns\TextColumn::make('updated_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\TernaryFilter::make('is_active')
                    ->label('Active Status')
                    ->trueLabel('Active Only')
                    ->falseLabel('Inactive Only'),
                
                Tables\Filters\TernaryFilter::make('is_best_value')
                    ->label('Best Value')
                    ->trueLabel('Best Value Only')
                    ->falseLabel('Not Best Value'),
            ])
            ->actions([
                Actions\EditAction::make(),
                Actions\Action::make('toggleStatus')
                    ->label(fn (TopUpOption $record) => $record->is_active ? 'Deactivate' : 'Activate')
                    ->color(fn (TopUpOption $record) => $record->is_active ? 'danger' : 'success')
                    ->action(function (TopUpOption $record) {
                        $record->update(['is_active' => !$record->is_active]);
                    })
                    ->requiresConfirmation(),
                Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Actions\BulkActionGroup::make([
                    Actions\DeleteBulkAction::make(),
                    Actions\BulkAction::make('activate')
                        ->action(fn ($records) => $records->each->update(['is_active' => true]))
                        ->requiresConfirmation(),
                    Actions\BulkAction::make('deactivate')
                        ->action(fn ($records) => $records->each->update(['is_active' => false]))
                        ->requiresConfirmation(),
                    Actions\BulkAction::make('markAsBestValue')
                        ->action(fn ($records) => $records->each->update(['is_best_value' => true]))
                        ->requiresConfirmation(),
                    Actions\BulkAction::make('unmarkAsBestValue')
                        ->action(fn ($records) => $records->each->update(['is_best_value' => false]))
                        ->requiresConfirmation(),
                ]),
            ])
            ->defaultSort('words', 'asc');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => ListTopUps::route('/'),
            'create' => CreateTopUp::route('/create'),
            'edit' => EditTopUp::route('/{record}/edit'),
        ];
    }
}