<?php

namespace App\Filament\Widgets;

use App\Models\Payment;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\DB;

class PaymentStats extends BaseWidget
{
    protected static ?int $sort = 8;
    protected function getStats(): array

    {
        $startDate = now()->startOfMonth()->format('Y-m-d');
        $endDate = now()->format('Y-m-d');

        return [
            Stat::make('Total Revenue', $this->getTotalRevenue($startDate, $endDate))
                ->description($this->getRevenueChange($startDate, $endDate))
                ->descriptionIcon($this->getRevenueChangeIcon($startDate, $endDate))
                ->color($this->getRevenueChangeColor($startDate, $endDate)),

            Stat::make('Successful Payments', $this->getSuccessfulPaymentsCount($startDate, $endDate))
                ->description($this->getPaymentSuccessRate($startDate, $endDate) . ' success rate')
                ->descriptionIcon('heroicon-o-check-badge')
                ->color('success'),

            Stat::make('Average Payment', $this->getAveragePayment($startDate, $endDate))
                ->description('Per successful transaction')
                ->descriptionIcon('heroicon-o-banknotes')
                ->color('info'),

            Stat::make('Top Agent', $this->getTopAgentRevenue($startDate, $endDate))
                ->description('Highest earning agent')
                ->descriptionIcon('heroicon-o-trophy')
                ->color('warning'),
        ];
    }

    private function getTotalRevenue($startDate, $endDate): string
    {
        $revenue = Payment::where('status', 'captured')
            ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->sum('amount');

        return '₹' . number_format($revenue / 100, 2);
    }

    private function getSuccessfulPaymentsCount($startDate, $endDate): string
    {
        return (string) Payment::where('status', 'captured')
            ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->count();
    }

    private function getPaymentSuccessRate($startDate, $endDate): string
    {
        $total = Payment::whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])->count();
        $successful = Payment::where('status', 'captured')
            ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->count();

        return $total > 0 ? round(($successful / $total) * 100, 1) . '%' : '0%';
    }

    private function getAveragePayment($startDate, $endDate): string
    {
        $avg = Payment::where('status', 'captured')
            ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->avg('amount');

        return '₹' . number_format(($avg ?? 0) / 100, 2);
    }

    private function getTopAgentRevenue($startDate, $endDate): string
    {
        $topAgent = Payment::where('status', 'captured')
            ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->select('travel_agent_id', DB::raw('SUM(amount) as total'))
            ->with('travelAgent')
            ->groupBy('travel_agent_id')
            ->orderBy('total', 'desc')
            ->first();

        return $topAgent ? ($topAgent->travelAgent->name ?? 'Unknown') : 'No data';
    }

    private function getRevenueChange($startDate, $endDate): string
    {
        $currentRevenue = Payment::where('status', 'captured')
            ->whereBetween('created_at', [$startDate, $endDate . ' 23:59:59'])
            ->sum('amount');

        $previousRevenue = Payment::where('status', 'captured')
            ->whereBetween('created_at', [
                now()->parse($startDate)->subMonth()->format('Y-m-d'),
                now()->parse($endDate)->subMonth()->format('Y-m-d') . ' 23:59:59'
            ])
            ->sum('amount');

        if ($previousRevenue == 0) return 'No previous data';

        $change = (($currentRevenue - $previousRevenue) / $previousRevenue) * 100;
        $direction = $change >= 0 ? 'up' : 'down';

        return abs(round($change, 1)) . '% ' . $direction;
    }

    private function getRevenueChangeIcon($startDate, $endDate): string
    {
        $change = $this->getRevenueChange($startDate, $endDate);
        return str_contains($change, 'up') ? 'heroicon-m-arrow-trending-up' : 'heroicon-m-arrow-trending-down';
    }

    private function getRevenueChangeColor($startDate, $endDate): string
    {
        $change = $this->getRevenueChange($startDate, $endDate);
        return str_contains($change, 'up') ? 'success' : 'danger';
    }
}