<?php

namespace App\Filament\Widgets;

use App\Models\Payment;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Widgets\TableWidget as BaseWidget;
use Illuminate\Support\Facades\DB;

class TopAgentsByRevenue extends BaseWidget
{
    protected static ?string $heading = 'Top Travel Agents by Revenue';
    
    protected int | string | array $columnSpan = 'full';
    protected static ?int $sort = 6;

    public function table(Table $table): Table
    {
        return $table
            ->query(
                Payment::with('travelAgent')
                    ->where('status', 'captured')
                    ->whereNotNull('travel_agent_id') // Add this line to filter out null agent IDs
                    ->select('travel_agent_id', DB::raw('SUM(amount) as total_revenue'), DB::raw('COUNT(*) as payment_count'))
                    ->groupBy('travel_agent_id')
                    ->orderBy('total_revenue', 'desc')
            )
            ->columns([
                Tables\Columns\TextColumn::make('travelAgent.name')
                    ->label('Travel Agent')
                    ->searchable()
                    ->sortable()
                    ->placeholder('N/A'), // Add placeholder for safety
                    
                Tables\Columns\TextColumn::make('total_revenue')
                    ->label('Total Revenue')
                    ->money('INR')
                    ->sortable()
                    ->alignCenter(),
                    
                Tables\Columns\TextColumn::make('payment_count')
                    ->label('Payments')
                    ->sortable()
                    ->alignCenter(),
                    
                Tables\Columns\TextColumn::make('avg_payment')
                    ->label('Avg Payment')
                    ->getStateUsing(fn ($record) => '₹' . number_format($record->payment_count > 0 ? $record->total_revenue / $record->payment_count : 0, 2))
                    ->alignCenter(),
            ])
            ->paginated(false);
    }

    // Changed to public to match parent class
    public function getTableRecordKey($record): string
    {
        return $record->travel_agent_id ?? 'unknown-' . uniqid();
    }
}