<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\TripPayment;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AppDashboardController extends Controller
{
    /**
     * Get total overview statistics (Widgets 1-4 + Cancellation metrics)
     */
    public function getTotalOverview(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            
            // Total Quotations (Widget 1)
            $totalQuotations = Invoice::where('travel_agent_id', $travelAgentId)
                ->quotations()
                ->count();

            // Total Billings (Widget 2)
            $totalBillings = Invoice::where('travel_agent_id', $travelAgentId)
                ->billings()
                ->count();

            // Total Pending Dues (Widget 3) - Only for Billing invoices, exclude Cancelled
            $totalPendingDues = Invoice::where('travel_agent_id', $travelAgentId)
                ->billings()
                ->where('status', '!=', 'Cancelled')
                ->where('remaining_amount', '>', 0)
                ->sum('remaining_amount');

            // Total Payments (Widget 4) - All successful payments
            $totalPayments = TripPayment::whereHas('invoice', function($query) use ($travelAgentId) {
                    $query->where('travel_agent_id', $travelAgentId);
                })
                ->successfulPayments()
                ->sum('amount');

            // Cancellation Metrics
            $totalCancelled = Invoice::where('travel_agent_id', $travelAgentId)
                ->cancelled()
                ->count();

            // Cancellation Rate: (Cancelled Billings / Total Billings) * 100
            $cancelledBillings = Invoice::where('travel_agent_id', $travelAgentId)
                ->billings()
                ->cancelled()
                ->count();
                
            $cancellationRate = $totalBillings > 0 
                ? ($cancelledBillings / $totalBillings) * 100 
                : 0;

            $todayCancelled = Invoice::where('travel_agent_id', $travelAgentId)
                ->cancelledToday()
                ->count();

            $monthCancelled = Invoice::where('travel_agent_id', $travelAgentId)
                ->cancelledThisMonth()
                ->count();

            // Lost Revenue from cancelled billings
            $lostRevenue = Invoice::where('travel_agent_id', $travelAgentId)
                ->billings()
                ->cancelled()
                ->sum('final_price');

            return response()->json([
                'success' => true,
                'data' => [
                    // Core Overview
                    'total_quotations' => $totalQuotations,
                    'total_billings' => $totalBillings,
                    'total_pending_dues' => (float) $totalPendingDues,
                    'total_payments' => (float) $totalPayments,
                    
                    // Cancellation Metrics
                    'total_cancelled' => $totalCancelled,
                    'cancellation_rate' => round($cancellationRate, 2),
                    'today_cancelled' => $todayCancelled,
                    'month_cancelled' => $monthCancelled,
                    'lost_revenue' => (float) $lostRevenue,
                    
                    // Formatted values for display
                    'formatted' => [
                        'total_pending_dues' => '₹' . number_format($totalPendingDues, 2),
                        'total_payments' => '₹' . number_format($totalPayments, 2),
                        'lost_revenue' => '₹' . number_format($lostRevenue, 2),
                        'cancellation_rate' => round($cancellationRate, 2) . '%',
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load dashboard data: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get today and tomorrow statistics (Widgets 5-11) - FIXED TOUR END CALCULATIONS
     */
    public function getTodayTomorrowStats(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $today = Carbon::today();
            $tomorrow = Carbon::tomorrow();

            // Today Quotation (Widget 5)
            $todayQuotation = Invoice::where('travel_agent_id', $travelAgentId)
                ->quotations()
                ->whereDate('created_at', $today)
                ->count();

            // Today Billing (Widget 6)
            $todayBilling = Invoice::where('travel_agent_id', $travelAgentId)
                ->billings()
                ->whereDate('created_at', $today)
                ->count();

            // Today Payment (Widget 7)
            $todayPayment = TripPayment::whereHas('invoice', function($query) use ($travelAgentId) {
                    $query->where('travel_agent_id', $travelAgentId);
                })
                ->successfulPayments()
                ->whereDate('payment_date', $today)
                ->sum('amount');

            // Today Trip Start (Widget 8)
            $todayTripStart = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereDate('tour_start_day', $today)
                ->where('status', '!=', 'Cancelled')
                ->where('type', '=', 'Billing')
                ->count();

            // Today Trip End (Widget 9) - FIXED CALCULATION: start_date + (duration_days - 1)
            $todayTripEnd = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereRaw('DATE_ADD(tour_start_day, INTERVAL (duration_days - 1) DAY) = ?', [$today])
                ->where('status', '!=', 'Cancelled')
                ->where('type', '=', 'Billing')
                ->count();

            // Tomorrow Trip Start (Widget 10)
            $tomorrowTripStart = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereDate('tour_start_day', $tomorrow)
                ->where('status', '!=', 'Cancelled')
                ->where('type', '=', 'Billing')
                ->count();

            // Tomorrow Trip End (Widget 11) - FIXED CALCULATION: start_date + (duration_days - 1)
            $tomorrowTripEnd = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereRaw('DATE_ADD(tour_start_day, INTERVAL (duration_days - 1) DAY) = ?', [$tomorrow])
                ->where('status', '!=', 'Cancelled')
                ->where('type', '=', 'Billing')
                ->count();

            return response()->json([
                'success' => true,
                'data' => [
                    'today_quotation' => $todayQuotation,
                    'today_billing' => $todayBilling,
                    'today_payment' => (float) $todayPayment,
                    'today_trip_start' => $todayTripStart,
                    'today_trip_end' => $todayTripEnd,
                    'tomorrow_trip_start' => $tomorrowTripStart,
                    'tomorrow_trip_end' => $tomorrowTripEnd,
                    
                    // Formatted values
                    'formatted' => [
                        'today_payment' => '₹' . number_format($todayPayment, 2),
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load today/tomorrow stats: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get pending dues detailed report
     */
    public function getPendingDuesReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;

            $invoices = Invoice::where('travel_agent_id', $travelAgentId)
                ->billings()
                ->where('status', '!=', 'Cancelled')
                ->where('remaining_amount', '>', 0)
                ->with('customer')
                ->orderBy('remaining_amount', 'desc')
                ->get()
                ->map(function ($invoice) {
                    return [
                        'invoice_id' => $invoice->id,
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'N/A',
                        'customer_mobile' => $invoice->customer ? $invoice->customer->mobile : 'N/A',
                        'package_name' => $invoice->package_name,
                        'tour_start_date' => $invoice->tour_start_day,
                        'tour_start_date_formatted' => $this->formatDateToReadable($invoice->tour_start_day),
                        'final_price' => (float) $invoice->final_price,
                        'paid_amount' => (float) $invoice->total_paid_amount,
                        'remaining_amount' => (float) $invoice->remaining_amount,
                        'payment_status' => $invoice->payment_status,
                        'formatted' => [
                            'final_price' => '₹' . number_format($invoice->final_price, 2),
                            'paid_amount' => '₹' . number_format($invoice->total_paid_amount, 2),
                            'remaining_amount' => '₹' . number_format($invoice->remaining_amount, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'pending_dues',
                    'total_records' => $invoices->count(),
                    'total_pending_amount' => $invoices->sum('remaining_amount'),
                    'invoices' => $invoices
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load pending dues report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Format date to human readable format (25th Sep 2025)
     */
    private function formatDateToReadable($dateString)
    {
        if (!$dateString) {
            return 'N/A';
        }

        try {
            $date = Carbon::parse($dateString);
            
            $day = $date->day;
            $month = $date->format('M');
            $year = $date->year;
            
            // Add ordinal suffix
            $ordinal = 'th';
            if ($day % 10 == 1 && $day != 11) {
                $ordinal = 'st';
            } elseif ($day % 10 == 2 && $day != 12) {
                $ordinal = 'nd';
            } elseif ($day % 10 == 3 && $day != 13) {
                $ordinal = 'rd';
            }
            
            return $day . $ordinal . ' ' . $month . ' ' . $year;
        } catch (\Exception $e) {
            return $dateString; // Return original if parsing fails
        }
    }

    /**
     * Get today's quotations detailed report
     */
    public function getTodayQuotationsReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $today = Carbon::today();

            $invoices = Invoice::where('travel_agent_id', $travelAgentId)
                ->quotations()
                ->whereDate('created_at', $today)
                ->with('customer')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($invoice) {
                    return [
                        'invoice_id' => $invoice->id,
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'N/A',
                        'customer_mobile' => $invoice->customer ? $invoice->customer->mobile : 'N/A',
                        'package_name' => $invoice->package_name,
                        'tour_start_date' => $invoice->tour_start_day,
                        'tour_start_date_formatted' => $this->formatDateToReadable($invoice->tour_start_day),
                        'final_price' => (float) $invoice->final_price,
                        'no_of_pax' => $invoice->no_of_pax,
                        'duration_days' => $invoice->duration_days,
                        'formatted' => [
                            'final_price' => '₹' . number_format($invoice->final_price, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'today_quotations',
                    'total_records' => $invoices->count(),
                    'total_amount' => $invoices->sum('final_price'),
                    'invoices' => $invoices
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load today quotations report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get today's billings detailed report
     */
    public function getTodayBillingsReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $today = Carbon::today();

            $invoices = Invoice::where('travel_agent_id', $travelAgentId)
                ->billings()
                ->whereDate('created_at', $today)
                ->with('customer')
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($invoice) {
                    return [
                        'invoice_id' => $invoice->id,
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'N/A',
                        'customer_mobile' => $invoice->customer ? $invoice->customer->mobile : 'N/A',
                        'package_name' => $invoice->package_name,
                        'tour_start_date' => $invoice->tour_start_day,
                        'tour_start_date_formatted' => $this->formatDateToReadable($invoice->tour_start_day),
                        'final_price' => (float) $invoice->final_price,
                        'paid_amount' => (float) $invoice->total_paid_amount,
                        'remaining_amount' => (float) $invoice->remaining_amount,
                        'payment_status' => $invoice->payment_status,
                        'formatted' => [
                            'final_price' => '₹' . number_format($invoice->final_price, 2),
                            'paid_amount' => '₹' . number_format($invoice->total_paid_amount, 2),
                            'remaining_amount' => '₹' . number_format($invoice->remaining_amount, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'today_billings',
                    'total_records' => $invoices->count(),
                    'total_amount' => $invoices->sum('final_price'),
                    'invoices' => $invoices
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load today billings report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get today's payments detailed report
     */
    public function getTodayPaymentsReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $today = Carbon::today();

            $payments = TripPayment::whereHas('invoice', function($query) use ($travelAgentId) {
                    $query->where('travel_agent_id', $travelAgentId);
                })
                ->successfulPayments()
                ->whereDate('payment_date', $today)
                ->with('invoice.customer')
                ->orderBy('payment_date', 'desc')
                ->get()
                ->map(function ($payment) {
                    return [
                        'invoice_id' => $payment->invoice_id,
                        'customer_name' => $payment->invoice->customer ? $payment->invoice->customer->name : 'N/A',
                        'customer_mobile' => $payment->invoice->customer ? $payment->invoice->customer->mobile : 'N/A',
                        'package_name' => $payment->invoice->package_name,
                        'payment_type' => $payment->type,
                        'amount' => (float) $payment->amount,
                        'payment_method' => $payment->payment_method,
                        'payment_date' => $payment->payment_date,
                        'formatted' => [
                            'amount' => '₹' . number_format($payment->amount, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'today_payments',
                    'total_records' => $payments->count(),
                    'total_amount' => $payments->sum('amount'),
                    'payments' => $payments
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load today payments report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get today's trip start detailed report
     */
    public function getTodayTripStartReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $today = Carbon::today();

            $invoices = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereDate('tour_start_day', $today)
                ->where('status', '!=', 'Cancelled')
                ->where('type', '=', 'Billing')
                ->with('customer')
                ->orderBy('tour_start_day', 'asc')
                ->get()
                ->map(function ($invoice) {
                    // Calculate tour end date: start_date + (duration_days - 1)
                    $startDate = Carbon::parse($invoice->tour_start_day);
                    $endDate = $startDate->copy()->addDays($invoice->duration_days - 1);
                    
                    return [
                        'invoice_id' => $invoice->id,
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'N/A',
                        'customer_mobile' => $invoice->customer ? $invoice->customer->mobile : 'N/A',
                        'package_name' => $invoice->package_name,
                        'tour_start_date' => $invoice->tour_start_day,
                        'tour_start_date_formatted' => $this->formatDateToReadable($invoice->tour_start_day),
                        'tour_end_date_formatted' => $this->formatDateToReadable($endDate),
                        'duration_days' => $invoice->duration_days,
                        'no_of_pax' => $invoice->no_of_pax,
                        'final_price' => (float) $invoice->final_price,
                        'formatted' => [
                            'final_price' => '₹' . number_format($invoice->final_price, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'today_trip_start',
                    'total_records' => $invoices->count(),
                    'invoices' => $invoices
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load today trip start report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get today's trip end detailed report - FIXED CALCULATION
     */
    public function getTodayTripEndReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $today = Carbon::today();

            $invoices = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereRaw('DATE_ADD(tour_start_day, INTERVAL (duration_days - 1) DAY) = ?', [$today])
                ->with('customer')
                ->orderBy('tour_start_day', 'asc')
                ->get()
                ->map(function ($invoice) {
                    // Calculate tour end date: start_date + (duration_days - 1)
                    $startDate = Carbon::parse($invoice->tour_start_day);
                    $endDate = $startDate->copy()->addDays($invoice->duration_days - 1);
                    
                    return [
                        'invoice_id' => $invoice->id,
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'N/A',
                        'customer_mobile' => $invoice->customer ? $invoice->customer->mobile : 'N/A',
                        'package_name' => $invoice->package_name,
                        'tour_start_date' => $invoice->tour_start_day,
                        'tour_start_date_formatted' => $this->formatDateToReadable($invoice->tour_start_day),
                        'tour_end_date_formatted' => $this->formatDateToReadable($endDate),
                        'duration_days' => $invoice->duration_days,
                        'no_of_pax' => $invoice->no_of_pax,
                        'final_price' => (float) $invoice->final_price,
                        'paid_amount' => (float) $invoice->total_paid_amount,
                        'remaining_amount' => (float) $invoice->remaining_amount,
                        'formatted' => [
                            'final_price' => '₹' . number_format($invoice->final_price, 2),
                            'paid_amount' => '₹' . number_format($invoice->total_paid_amount, 2),
                            'remaining_amount' => '₹' . number_format($invoice->remaining_amount, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'today_trip_end',
                    'total_records' => $invoices->count(),
                    'invoices' => $invoices
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load today trip end report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tomorrow's trip start detailed report
     */
    public function getTomorrowTripStartReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $tomorrow = Carbon::tomorrow();

            $invoices = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereDate('tour_start_day', $tomorrow)
                ->with('customer')
                ->orderBy('tour_start_day', 'asc')
                ->get()
                ->map(function ($invoice) {
                    // Calculate tour end date: start_date + (duration_days - 1)
                    $startDate = Carbon::parse($invoice->tour_start_day);
                    $endDate = $startDate->copy()->addDays($invoice->duration_days - 1);
                    
                    return [
                        'invoice_id' => $invoice->id,
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'N/A',
                        'customer_mobile' => $invoice->customer ? $invoice->customer->mobile : 'N/A',
                        'package_name' => $invoice->package_name,
                        'tour_start_date' => $invoice->tour_start_day,
                        'tour_start_date_formatted' => $this->formatDateToReadable($invoice->tour_start_day),
                        'tour_end_date_formatted' => $this->formatDateToReadable($endDate),
                        'duration_days' => $invoice->duration_days,
                        'no_of_pax' => $invoice->no_of_pax,
                        'final_price' => (float) $invoice->final_price,
                        'formatted' => [
                            'final_price' => '₹' . number_format($invoice->final_price, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'tomorrow_trip_start',
                    'total_records' => $invoices->count(),
                    'invoices' => $invoices
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load tomorrow trip start report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get tomorrow's trip end detailed report - FIXED CALCULATION
     */
    public function getTomorrowTripEndReport(Request $request)
    {
        try {
            $travelAgentId = auth()->user()->travel_agent_id;
            $tomorrow = Carbon::tomorrow();

            $invoices = Invoice::where('travel_agent_id', $travelAgentId)
                ->whereRaw('DATE_ADD(tour_start_day, INTERVAL (duration_days - 1) DAY) = ?', [$tomorrow])
                ->with('customer')
                ->orderBy('tour_start_day', 'asc')
                ->get()
                ->map(function ($invoice) {
                    // Calculate tour end date: start_date + (duration_days - 1)
                    $startDate = Carbon::parse($invoice->tour_start_day);
                    $endDate = $startDate->copy()->addDays($invoice->duration_days - 1);
                    
                    return [
                        'invoice_id' => $invoice->id,
                        'customer_name' => $invoice->customer ? $invoice->customer->name : 'N/A',
                        'customer_mobile' => $invoice->customer ? $invoice->customer->mobile : 'N/A',
                        'package_name' => $invoice->package_name,
                        'tour_start_date' => $invoice->tour_start_day,
                        'tour_start_date_formatted' => $this->formatDateToReadable($invoice->tour_start_day),
                        'tour_end_date_formatted' => $this->formatDateToReadable($endDate),
                        'duration_days' => $invoice->duration_days,
                        'no_of_pax' => $invoice->no_of_pax,
                        'final_price' => (float) $invoice->final_price,
                        'paid_amount' => (float) $invoice->total_paid_amount,
                        'remaining_amount' => (float) $invoice->remaining_amount,
                        'formatted' => [
                            'final_price' => '₹' . number_format($invoice->final_price, 2),
                            'paid_amount' => '₹' . number_format($invoice->total_paid_amount, 2),
                            'remaining_amount' => '₹' . number_format($invoice->remaining_amount, 2),
                        ]
                    ];
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'report_type' => 'tomorrow_trip_end',
                    'total_records' => $invoices->count(),
                    'invoices' => $invoices
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load tomorrow trip end report: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get monthly charts data (Widgets 13-14)
     */
    public function getMonthlyCharts(Request $request)
    {
        // Will implement in next step
        return response()->json([
            'success' => true,
            'data' => [
                'message' => 'Endpoint under development'
            ]
        ]);
    }
}