<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\TravelAgent; // <-- Added for eager loading the relationship
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;
use Illuminate\Validation\ValidationException; // Added for validation

// NEW: Firebase Admin SDK Imports
use Kreait\Firebase\Contract\Auth; 
use Kreait\Firebase\Exception\Auth\FailedToVerifyToken;

class AuthController extends Controller
{
    // NEW: Property to hold the Firebase Auth instance
    protected $firebaseAuth;

    // NEW: Constructor to inject the Firebase Auth instance
    public function __construct(Auth $firebaseAuth)
    {
        $this->firebaseAuth = $firebaseAuth;
    }

// ----------------------------------------------------------------------
// NEW FIREBASE AUTHENTICATION METHOD
// ----------------------------------------------------------------------

    /**
     * Verify Firebase ID Token and Log in/Register the User.
     * The Flutter client sends the Firebase ID Token (JWT) after successful 
     * phone number verification.
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function firebaseAuth(Request $request)
    {
        // 1. Validate the incoming request data
        $request->validate([
            'id_token' => 'required|string',
            'mobile' => 'required|string',
        ]);

        $idToken = $request->id_token;
        $mobile = $request->mobile;
        
        try {
            // 2. VERIFY the ID Token using the Firebase Admin SDK
            $verifiedIdToken = $this->firebaseAuth->verifyIdToken($idToken);
            $firebaseUid = $verifiedIdToken->claims()->get('sub');

            // 3. Find the user by mobile number
            $user = User::where('mobile', $mobile)->first();

            if (!$user) {
                // This case should ideally not happen if admin pre-registers users.
                // If it happens, it means the user is trying to log in with a mobile 
                // not associated with any travel agent.
                return response()->json([
                    'message' => 'User not registered under any travel agent'
                ], 404);
            }

            // 4. Check if the user is active
            if (!$user->is_active) {
                 return response()->json([
                    'message' => 'Your account is currently deactivated. Please contact your administrator.'
                ], 403);
            }

            // 5. User Found: Update firebase_uid if null (First-Time Firebase login)
            if (is_null($user->firebase_uid)) {
                $user->firebase_uid = $firebaseUid;
                $user->last_login_method = 'Firebase OTP - First Time';
            } else {
                // Check if the UID matches to prevent mobile squatting
                if ($user->firebase_uid !== $firebaseUid) {
                     return response()->json([
                        'message' => 'Security token mismatch. Please contact support.'
                    ], 403);
                }
                $user->last_login_method = 'Firebase OTP - Repeat';
            }

            // 6. Update last login details
            $user->last_login_at = now();
            // Extend access to cover the time before PIN expires
            $user->access_expires_at = now()->addDays(30); 
            $user->save();

            // 7. Create API Token
            $token = $user->createToken('api-token')->plainTextToken;

            $this->notifyTravelAgent($user, 'Firebase OTP');

            return response()->json([
                // Only return necessary user info. pin_code determines the flow.
                'user' => $user->only(['id', 'name', 'email', 'mobile', 'travel_agent_id', 'pin_code']),
                'token' => $token
            ]);

        } catch (FailedToVerifyToken $e) {
            return response()->json(['message' => 'Invalid or expired ID token.', 'error' => $e->getMessage()], 401);
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('Firebase Auth Error: ' . $e->getMessage());
            return response()->json(['message' => 'Authentication failed due to an internal error.'], 500);
        }
    }
// ----------------------------------------------------------------------
// NEW PIN SETUP METHOD (To be called after successful first-time firebaseAuth)
// ----------------------------------------------------------------------

    /**
     * Allows a successfully authenticated user to set their PIN for the first time.
     * Must be authenticated via Sanctum (i.e., have an API token from firebaseAuth).
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function setPin(Request $request)
    {
        $request->validate([
            'pin' => 'required|string|min:4|max:6', // Validate PIN format
        ]);

        /** @var \App\Models\User $user */
        $user = $request->user();

        // 1. Check if PIN is already set
        if (!is_null($user->pin_code)) {
            throw ValidationException::withMessages([
                'pin' => ['A PIN is already set for this account. Use the change PIN feature if needed.'],
            ]);
        }
        
        // 2. Check if the user is active
        if (!$user->is_active) {
             return response()->json([
                'message' => 'Your account is currently deactivated. Please contact your administrator.'
            ], 403);
        }

        // 3. Hash and save the new PIN
        $user->pin_code = Hash::make($request->pin);
        $user->save();

        return response()->json([
            'message' => 'PIN set successfully. You can now use your PIN for quick login.',
        ], 200);
    }
    
// ----------------------------------------------------------------------
// LOGIN WITH PIN METHOD
// ----------------------------------------------------------------------

    /**
     * Login using PIN code.
     */
    public function loginWithPin(Request $request)
    {
        $request->validate([
            'mobile' => 'required',
            'pin' => 'required|string|min:4|max:6'
        ]);

        $user = User::where('mobile', $request->mobile)
                    ->first();

        if (!$user || is_null($user->pin_code) || !Hash::check($request->pin, $user->pin_code)) {
            return response()->json(['message' => 'Invalid mobile or PIN code.'], 401);
        }

        if (!$user->is_active) {
             return response()->json([
                'message' => 'Your account is currently deactivated. Please contact your administrator.'
            ], 403);
        }

        // Check PIN expiration (e.g., 30 days)
        if ($user->access_expires_at && $user->access_expires_at->isPast()) {
            // Log the user out remotely (or client-side) and force OTP re-login
            $user->tokens()->delete();
             return response()->json([
                'message' => 'Your PIN has expired. Please login with OTP again.',
                'pin_expired' => true
            ], 401);
        }

        $user->last_login_at = now();
        $user->last_login_method = 'PIN Code';
        $user->save();

        // Create new token for the user
        $token = $user->createToken('api-token')->plainTextToken;

        $this->notifyTravelAgent($user, 'PIN Code');

        return response()->json([
            'user' => $user->only(['id', 'name', 'email', 'mobile', 'travel_agent_id', 'pin_code']),
            'token' => $token
        ]);
    }

// ----------------------------------------------------------------------
// PASSWORD AND OTHER METHODS (UNCHANGED/REMOVED FOR BREVITY)
// ----------------------------------------------------------------------
    
    /**
     * Login using traditional email/password. (Keeping this optional)
     */
    public function loginWithPassword(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        $user = User::where('email', $request->email)->first();
        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json(['message' => 'Invalid credentials'], 401);
        }

        if (!$user->is_active) {
             return response()->json([
                'message' => 'Your account is currently deactivated. Please contact your administrator.'
            ], 403);
        }

        $user->last_login_at = now();
        $user->last_login_method = 'Password';
        $user->access_expires_at = now()->addDays(30);
        $user->save();

        $token = $user->createToken('api-token')->plainTextToken;

        $this->notifyTravelAgent($user, 'Password');

        return response()->json([
            'user' => $user->only(['id', 'name', 'email', 'mobile', 'travel_agent_id', 'pin_code']),
            'token' => $token
        ]);
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();
        return response()->json(['message'=>'Logged out']);
    }

    public function profile(Request $request)
    {
        $user = $request->user()->load('travelAgent'); // Eager load the relationship

        return response()->json([
            'id' => $user->id,
            'name' => $user->name,
            'email' => $user->email,
            'mobile' => $user->mobile,
            'travel_agent_id' => $user->travel_agent_id,
            'pin_code' => $user->pin_code,
            'company_name' => $user->travelAgent->name ?? null, 
        ]);
    }

    /**
     * Notify Travel Agent (DEV: Placeholder for real notification)
     * This is a simple placeholder method and should be replaced with actual 
     * notification logic (e.g., queuing a job, sending SMS/email).
     */
    private function notifyTravelAgent($user, $method)
    {
        // Placeholder for notification logic
        // E.g., Log or send an internal email/SMS
        \Log::info("User {$user->mobile} logged in via {$method} for agent {$user->travel_agent_id}");
    }
}