<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\DayDescription;

class DayDescriptionController extends Controller
{
    /**
     * Display a listing of day descriptions
     */
    public function index()
    {
        $descriptions = DayDescription::with('travelAgent')->get();
        return response()->json($descriptions);
    }

    /**
     * Store a newly created day description
     */
    public function store(Request $request)
    {
        $request->validate([
            'travel_agent_id' => 'required|exists:travel_agents,id',
            'daydescription_title' => 'required|string|max:255',
            'daydescription_description' => 'required|string'
        ]);

        $description = DayDescription::create($request->all());

        return response()->json($description, 201);
    }

    /**
     * Display the specified day description
     */
    public function show(string $id)
    {
        $description = DayDescription::with('travelAgent')->findOrFail($id);
        return response()->json($description);
    }

    /**
     * Update the specified day description
     */
    public function update(Request $request, string $id)
    {
        $description = DayDescription::findOrFail($id);

        $request->validate([
            'travel_agent_id' => 'sometimes|exists:travel_agents,id',
            'daydescription_title' => 'sometimes|string|max:255',
            'daydescription_description' => 'sometimes|string'
        ]);

        $description->update($request->all());

        return response()->json($description);
    }

    /**
     * Remove the specified day description
     */
    public function destroy(string $id)
    {
        $description = DayDescription::findOrFail($id);
        $description->delete();

        return response()->json(['message' => 'Day description deleted successfully']);
    }

    /**
     * Search day descriptions by title
     */
    public function search(Request $request)
    {
        $request->validate([
            'query' => 'required|string'
        ]);

        $descriptions = DayDescription::where('daydescription_title', 'like', '%' . $request->query . '%')
            ->orWhere('daydescription_description', 'like', '%' . $request->query . '%')
            ->get();

        return response()->json($descriptions);
    }
}