<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\TravelAgent;
use App\Models\AiPlan;
use App\Models\AiSubscription;
use App\Models\AiKey;
use App\Models\TopUpOption; // ADDED: TopUpOption model
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class SubscriptionController extends Controller
{
    /**
     * Subscribe/Change plan (TEST MODE - No payment required)
     */
    public function subscribe(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $validator = Validator::make($request->all(), [
                'travel_agent_id' => 'required|exists:travel_agents,id',
                'plan_id' => 'required|exists:ai_plans,id',
            ]);

            if ($validator->fails()) {
                return response()->json(['error' => 'Validation failed', 'messages' => $validator->errors()], 422);
            }

            $agentId = $request->input('travel_agent_id');
            $newPlanId = $request->input('plan_id');
            $newPlan = AiPlan::find($newPlanId);

            if (!$newPlan) {
                return response()->json(['error' => 'Plan not found.'], 404);
            }

            // Get current active subscription
            $currentSubscription = AiSubscription::where('travel_agent_id', $agentId)
                ->where('status', 'active')
                ->first();

            // If already subscribed to same plan
            if ($currentSubscription && $currentSubscription->plan_id == $newPlanId) {
                return response()->json([
                    'message' => 'You are already subscribed to this plan.',
                    'current_plan' => $newPlan->name
                ], 409);
            }

            // Deactivate current subscription if exists - FIXED: Use 'canceled' (one L)
            if ($currentSubscription) {
                Log::info("Changing plan from {$currentSubscription->plan->name} to {$newPlan->name} for Agent ID: {$agentId}");
                
                $currentSubscription->update([
                    'status' => 'canceled', // ✅ FIX: Changed from 'cancelled' to 'canceled'
                    'end_date' => now()
                ]);
            } else {
                Log::info("New subscription to {$newPlan->name} for Agent ID: {$agentId}");
            }

            // Create new subscription
            $subscriptionData = [
                'travel_agent_id' => $agentId,
                'plan_id' => $newPlanId,
                'status' => 'active',
                'start_date' => now(),
                'end_date' => now()->addMonth(),
                'billing_cycle_start' => now(),
                'billing_cycle_end' => now()->addMonth(),
                'words_remaining' => $newPlan->monthly_words,
                'words_used' => 0,
                'price' => $newPlan->price,
                'payment_status' => 'succeeded',
                'payment_id' => 'test_mode_' . uniqid(),
            ];

            Log::info('Creating subscription with data:', $subscriptionData);

            $newSubscription = AiSubscription::create($subscriptionData);

            // Update AI key subscription dates
            AiKey::where('travel_agent_id', $agentId)->update([
                'subscription_start_date' => now(),
                'subscription_end_date' => now()->addMonth(),
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Plan activated successfully!',
                'plan' => $newPlan->name,
                'monthly_words' => $newPlan->monthly_words,
                'price' => $newPlan->price,
                'subscription' => $newSubscription,
                'test_mode' => true
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Subscription error: ' . $e->getMessage());
            
            return response()->json([
                'error' => 'Subscription failed',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all available plans
     */
    public function getPlans()
    {
        $plans = AiPlan::where('is_active', true)->orderBy('price')->get()->map(function ($plan) {
            return [
                'id' => $plan->id,
                'name' => $plan->name,
                'display_name' => $plan->display_name,
                'monthly_words' => $plan->monthly_words,
                'total_words' => $plan->monthly_words + $plan->free_words_included,
                'free_words_included' => $plan->free_words_included,
                'price' => (float) $plan->price, // ✅ Convert to float
                'top_up_rate' => (float) $plan->top_up_rate, // ✅ Convert to float
                'carry_forward_months' => $plan->carry_forward_months,
                'description' => $plan->description,
                'features' => $plan->features,
                'is_free' => $plan->name === 'free',
            ];
        });

        return response()->json(['plans' => $plans]);
    }

    /**
     * Get all available top-up options.
     * NEW METHOD
     */
    public function getTopUpOptions()
    {
        $topUps = TopUpOption::where('is_active', true)->orderBy('price')->get();
        return response()->json(['top_ups' => $topUps]);
    }

    /**
     * Get current subscription details
     */
    public function getCurrentPlan(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'travel_agent_id' => 'required|exists:travel_agents,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Validation failed', 'messages' => $validator->errors()], 422);
        }

        $agentId = $request->travel_agent_id;
        
        $subscription = AiSubscription::where('travel_agent_id', $agentId)
            ->where('status', 'active')
            ->with('plan')
            ->first();

        $currentPlan = $subscription ? $subscription->plan : AiPlan::where('name', 'free')->first();

        return response()->json([
            'current_plan' => $currentPlan,
            'subscription' => $subscription,
            'is_trial' => !$subscription,
            'words_remaining' => $subscription ? $subscription->words_remaining : 0,
            'words_used' => $subscription ? $subscription->words_used : 0,
        ]);
    }

    /**
     * Cancel subscription - FIXED: Use 'canceled' (one L)
     */
    public function cancelSubscription(Request $request)
    {
        DB::beginTransaction();
        
        try {
            $validator = Validator::make($request->all(), [
                'travel_agent_id' => 'required|exists:travel_agents,id'
            ]);

            if ($validator->fails()) {
                return response()->json(['error' => 'Validation failed'], 422);
            }

            $agentId = $request->travel_agent_id;
            
            $subscription = AiSubscription::where('travel_agent_id', $agentId)
                ->where('status', 'active')
                ->first();

            if (!$subscription) {
                return response()->json(['error' => 'No active subscription found'], 404);
            }

            $planName = $subscription->plan->name;
            
            $subscription->update([
                'status' => 'canceled', // ✅ FIX: Changed from 'cancelled' to 'canceled'
                'end_date' => now()
            ]);

            // Reset AI key subscription dates
            AiKey::where('travel_agent_id', $agentId)->update([
                'subscription_end_date' => now(),
            ]);

            DB::commit();

            return response()->json([
                'message' => 'Subscription cancelled successfully. Reverted to Free plan.',
                'cancelled_plan' => $planName,
                'free_words_available' => 500
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['error' => 'Cancellation failed: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get usage statistics
     */
    public function getUsage(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'travel_agent_id' => 'required|exists:travel_agents,id'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Validation failed'], 422);
        }

        $agentId = $request->travel_agent_id;
        $aiKey = AiKey::where('travel_agent_id', $agentId)->first();

        if (!$aiKey) {
            return response()->json(['error' => 'AI key not found'], 404);
        }

        $subscription = AiSubscription::where('travel_agent_id', $agentId)
            ->where('status', 'active')
            ->with('plan')
            ->first();

        // Calculate usage
        $freeWordsUsed = $aiKey->free_words_used ?? 0;
        $freeWordsRemaining = max(0, 500 - $freeWordsUsed);
        
        $subscriptionWordsUsed = $subscription ? $subscription->words_used : 0;
        $subscriptionWordsRemaining = $subscription ? $subscription->words_remaining : 0;
        $subscriptionTotalWords = $subscription ? $subscription->plan->monthly_words : 0;

        return response()->json([
            'free_words' => [
                'used' => $freeWordsUsed,
                'remaining' => $freeWordsRemaining,
                'total' => 500
            ],
            'subscription_words' => [
                'used' => $subscriptionWordsUsed,
                'remaining' => $subscriptionWordsRemaining,
                'total' => $subscriptionTotalWords
            ],
            'total_words_used' => $aiKey->total_words_used ?? 0,
            'has_active_subscription' => (bool) $subscription,
            'current_plan' => $subscription ? $subscription->plan->name : 'free'
        ]);
    }

    /**
     * Quick fix: Update all existing 'cancelled' to 'canceled'
     */
    public function fixStatusSpelling()
    {
        try {
            $updated = AiSubscription::where('status', 'cancelled')->update(['status' => 'canceled']);
            
            return response()->json([
                'message' => 'Status spelling fixed',
                'records_updated' => $updated
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Fix failed: ' . $e->getMessage()], 500);
        }
    }
}