<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\SuperAdmin;
use App\Models\SuperAdminLoginLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $superadmin = SuperAdmin::where('email', $request->email)->first();

        // Log login attempt
        $loginLog = $this->createLoginLog($superadmin, $request, false);

        // Check if account exists and is active
        if (!$superadmin) {
            $loginLog->update([
                'login_successful' => false,
                'failure_reason' => 'Account not found'
            ]);
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        if (!$superadmin->is_active) {
            $loginLog->update([
                'login_successful' => false,
                'failure_reason' => 'Account deactivated'
            ]);
            throw ValidationException::withMessages([
                'email' => ['Your account has been deactivated.'],
            ]);
        }

        if ($superadmin->isLocked()) {
            $loginLog->update([
                'login_successful' => false,
                'failure_reason' => 'Account temporarily locked'
            ]);
            throw ValidationException::withMessages([
                'email' => ['Account temporarily locked. Try again later.'],
            ]);
        }

        // Verify password
        if (!Hash::check($request->password, $superadmin->password)) {
            $superadmin->incrementFailedAttempts();
            $loginLog->update([
                'login_successful' => false,
                'failure_reason' => 'Invalid password'
            ]);
            throw ValidationException::withMessages([
                'email' => ['The provided credentials are incorrect.'],
            ]);
        }

        // Successful login
        $superadmin->resetFailedAttempts();
        $superadmin->update([
            'last_login_ip' => $request->ip(),
            'last_login_at' => now(),
        ]);

        $token = $superadmin->createToken('superadmin-token')->plainTextToken;

        // Update login log with success
        $loginLog->update([
            'login_successful' => true,
            'session_id' => Str::random(40),
        ]);

        return response()->json([
            'token' => $token,
            'superadmin' => $superadmin,
            'login_log_id' => $loginLog->id,
        ]);
    }

    public function logout(Request $request)
    {
        // Update login log with logout time
        if ($request->login_log_id) {
            SuperAdminLoginLog::where('id', $request->login_log_id)
                ->update(['logout_at' => now()]);
        }

        $request->user()->currentAccessToken()->delete();
        
        return response()->json(['message' => 'Logged out successfully']);
    }

    public function me(Request $request)
    {
        $superadmin = $request->user()->load(['loginLogs' => function($query) {
            $query->orderBy('login_at', 'desc')->limit(10);
        }]);

        return response()->json($superadmin);
    }

    public function loginHistory(Request $request)
    {
        $logs = SuperAdminLoginLog::with('superAdmin')
            ->orderBy('login_at', 'desc')
            ->paginate(20);

        return response()->json($logs);
    }

    private function createLoginLog($superadmin, Request $request, $success = false)
    {
        return SuperAdminLoginLog::create([
            'super_admin_id' => $superadmin?->id,
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'login_at' => now(),
            'login_successful' => $success,
            'failure_reason' => $success ? null : 'Authentication failed',
        ]);
    }
}