<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Http\Controllers\Controller;
use App\Models\TravelAgent;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class TravelAgentController extends Controller
{
    public function index(Request $request)
    {
        $query = TravelAgent::withCount(['users', 'invoices', 'aiSubscriptions']);

        // Search filter
        if ($request->has('search') && $request->search) {
            $query->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
        }

        // Status filter
        if ($request->has('status') && in_array($request->status, ['active', 'inactive'])) {
            $query->where('is_active', $request->status === 'active');
        }

        $agents = $query->orderBy('created_at', 'desc')->paginate(20);

        return response()->json($agents);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:travel_agents,email',
            'phone' => 'nullable|string',
            'contact_person_name' => 'nullable|string',
            'gst_number' => 'nullable|string',
            'address' => 'nullable|string',
        ]);

        $travelAgent = TravelAgent::create($validated);

        return response()->json([
            'message' => 'Travel Agent created successfully',
            'data' => $travelAgent
        ], 201);
    }

    public function show($id)
    {
        $travelAgent = TravelAgent::with(['users', 'aiSubscriptions.plan'])
            ->withCount(['users', 'invoices', 'aiUsageLogs'])
            ->findOrFail($id);

        return response()->json($travelAgent);
    }

    public function update(Request $request, $id)
    {
        $travelAgent = TravelAgent::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:travel_agents,email,' . $travelAgent->id,
            'phone' => 'nullable|string',
            'contact_person_name' => 'nullable|string',
            'gst_number' => 'nullable|string',
            'address' => 'nullable|string',
            'is_active' => 'sometimes|boolean',
        ]);

        $travelAgent->update($validated);

        return response()->json([
            'message' => 'Travel Agent updated successfully',
            'data' => $travelAgent
        ]);
    }

    public function toggleStatus($id)
    {
        $travelAgent = TravelAgent::findOrFail($id);
        $travelAgent->update(['is_active' => !$travelAgent->is_active]);

        $action = $travelAgent->is_active ? 'activated' : 'deactivated';

        return response()->json([
            'message' => "Travel Agent {$action} successfully",
            'data' => $travelAgent
        ]);
    }

    public function getAgentUsers($id)
    {
        $users = User::where('travel_agent_id', $id)
            ->with('travelAgent')
            ->orderBy('is_admin', 'desc')
            ->orderBy('name')
            ->paginate(20);

        return response()->json($users);
    }

    public function getAgentStats($id)
    {
        $travelAgent = TravelAgent::withCount([
            'users', 
            'invoices',
            'aiUsageLogs as total_ai_usage' => function($query) {
                $query->select(\DB::raw('SUM(word_count)'));
            }
        ])->findOrFail($id);

        $revenue = \App\Models\Payment::where('travel_agent_id', $id)
            ->where('status', 'captured')
            ->sum('amount');

        $activeSubscription = $travelAgent->activeAiSubscription();

        return response()->json([
            'total_users' => $travelAgent->users_count,
            'total_invoices' => $travelAgent->invoices_count,
            'total_ai_usage' => $travelAgent->total_ai_usage ?? 0,
            'total_revenue' => (float) $revenue,
            'active_subscription' => $activeSubscription,
        ]);
    }
}
