<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\TripPayment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TripPaymentController extends Controller
{
    /**
     * Get all trip payments for an invoice.
     */
    public function index(Invoice $invoice)
    {
        $payments = $invoice->tripPayments()->orderBy('payment_date', 'desc')->get();

        return response()->json([
            'success' => true,
            'payments' => $payments,
            'payment_summary' => $invoice->getPaymentSummary(),
        ]);
    }

    /**
     * Record a new trip payment.
     */
    public function store(Request $request, Invoice $invoice)
    {
        $currentTotalPaid = $invoice->calculateTotalPaid();
        $maxPaymentAmount = $invoice->final_price - $currentTotalPaid;

        $request->validate([
            'type' => 'required|in:Advance,Installment,Full',
            'amount' => [
                'required',
                'numeric',
                'min:0.01',
                function ($attribute, $value, $fail) use ($maxPaymentAmount) {
                    if ($value > round($maxPaymentAmount, 2)) {
                        $fail("Payment amount cannot exceed remaining balance of ₹" . number_format($maxPaymentAmount, 2));
                    }
                },
            ],
            'payment_method' => 'required|string|max:50',
            'transaction_id' => 'nullable|string|max:100',
            'reference_number' => 'nullable|string|max:100',
            'payment_date' => 'required|date',
            'notes' => 'nullable|string|max:500',
        ]);

        try {
            $payment = DB::transaction(function () use ($request, $invoice) {
                $lockedInvoice = Invoice::where('id', $invoice->id)->lockForUpdate()->first();
                $currentPaid = $lockedInvoice->calculateTotalPaid();
                $remaining = $lockedInvoice->final_price - $currentPaid;

                if ($request->amount > round($remaining, 2)) {
                    throw new \Exception('Payment amount exceeds current remaining balance.');
                }

                $payment = TripPayment::create([
                    'invoice_id' => $invoice->id,
                    'type' => $request->type,
                    'amount' => $request->amount,
                    'payment_method' => $request->payment_method,
                    'transaction_id' => $request->transaction_id,
                    'reference_number' => $request->reference_number,
                    'payment_date' => $request->payment_date,
                    'notes' => $request->notes,
                    'status' => 'Completed',
                ]);

                // Optional: update invoice summary if you have a method
                if (method_exists($lockedInvoice, 'updatePaymentSummary')) {
                    $lockedInvoice->updatePaymentSummary();
                }

                return $payment;
            });

            return response()->json([
                'success' => true,
                'message' => 'Payment recorded successfully',
                'payment' => $payment,
                'payment_summary' => $invoice->fresh()->getPaymentSummary(),
            ]);
        } catch (\Exception $e) {
            Log::error('Trip payment recording failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to record payment: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Process a refund for a trip.
     */
    public function processRefund(Request $request, Invoice $invoice)
    {
        $totalPaid = $invoice->calculateTotalPaid();
        $alreadyRefunded = $invoice->refunds()->sum('amount');
        $maxRefundable = $totalPaid - $alreadyRefunded;

        $request->validate([
            'refund_amount' => [
                'required',
                'numeric',
                'min:0.01',
                function ($attribute, $value, $fail) use ($maxRefundable) {
                    if ($value > round($maxRefundable, 2)) {
                        $fail('Refund amount cannot exceed refundable amount of ₹' . number_format($maxRefundable, 2));
                    }
                },
            ],
            'payment_method' => 'required|string|max:50',
            'transaction_id' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:500',
        ]);

        try {
            $refund = DB::transaction(function () use ($request, $invoice) {
                $lockedInvoice = Invoice::where('id', $invoice->id)->lockForUpdate()->first();

                $currentTotalPaid = $lockedInvoice->calculateTotalPaid();
                $currentRefunded = $lockedInvoice->refunds()->sum('amount');
                $currentRefundable = $currentTotalPaid - $currentRefunded;

                if ($request->refund_amount > round($currentRefundable, 2)) {
                    throw new \Exception('Refund amount exceeds available refundable amount.');
                }

                $refund = TripPayment::create([
                    'invoice_id' => $invoice->id,
                    'type' => 'Refund',
                    'amount' => $request->refund_amount,
                    'payment_method' => $request->payment_method,
                    'transaction_id' => $request->transaction_id,
                    'reference_number' => 'REFUND_' . time(),
                    'payment_date' => now(),
                    'notes' => ($request->notes ?? '') . ' | Refund processed for cancelled trip',
                    'status' => 'Completed',
                ]);

                // Update invoice refundable amount and status
                $newRefundable = max(0, $currentRefundable - $request->refund_amount);
                $lockedInvoice->update([
                    'refundable_amount' => $newRefundable,
                    'refund_status' => $newRefundable > 0 ? 'Partially Refunded' : 'Fully Refunded',
                ]);

                return $refund;
            });

            return response()->json([
                'success' => true,
                'message' => 'Refund processed successfully',
                'refund' => $refund,
            ]);
        } catch (\Exception $e) {
            Log::error('Trip refund processing failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to process refund: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get payment statistics for an invoice.
     */
    public function getPaymentStats(Invoice $invoice)
    {
        $stats = [
            'total_advance' => $invoice->advancePayments()->sum('amount'),
            'total_installments' => $invoice->installmentPayments()->sum('amount'),
            'total_refunds' => $invoice->refunds()->sum('amount'),
            'payment_history' => $invoice->getPaymentHistory(),
        ];

        return response()->json([
            'success' => true,
            'stats' => $stats,
        ]);
    }
}
