<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AiSubscription extends Model
{
    use HasFactory;

    protected $fillable = [
        'travel_agent_id',
        'plan_id',
        'words_used',
        'words_remaining',
        'billing_cycle_start',
        'billing_cycle_end',
        'start_date',
        'end_date',
        'payment_status',
        'payment_id',
        'status',
        'stripe_subscription_id',
        'stripe_price_id',
        'price'
    ];

    protected $casts = [
        'billing_cycle_start' => 'datetime',
        'billing_cycle_end' => 'datetime',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'words_used' => 'integer',
        'words_remaining' => 'integer',
        'price' => 'decimal:2'
    ];

    // ✅ FIX: Use correct status values
    const STATUS_ACTIVE = 'active';
    const STATUS_CANCELED = 'canceled'; // One 'L'
    const STATUS_EXPIRED = 'expired';
    const STATUS_PENDING = 'pending';

    // ✅ FIX: Scope for active subscriptions
    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE)
                    ->where('end_date', '>', now());
    }

    // Relationship with TravelAgent
    public function travelAgent()
    {
        return $this->belongsTo(TravelAgent::class);
    }

    // Relationship with Plan
    public function plan()
    {
        return $this->belongsTo(AiPlan::class);
    }

    // Check if subscription is active
    public function isActive()
    {
        return $this->status === self::STATUS_ACTIVE && 
               $this->end_date && 
               $this->end_date->isFuture();
    }

    // Increment words used
    public function incrementWordsUsed($wordCount)
    {
        $this->increment('words_used', $wordCount);
        $this->decrement('words_remaining', $wordCount);
    }

    // ✅ ADD THIS MISSING METHOD:
    /**
     * Get remaining words for the subscription
     */
    public function getRemainingWords()
    {
        return $this->words_remaining;
    }

    // ✅ OPTIONAL: You might also need this method if your AiKey model calls it
    /**
     * Get total available words including plan words and top-ups
     */
    public function getTotalAvailableWords()
    {
        $planWords = $this->plan ? $this->plan->total_words : 0;
        
        // If you have top-ups, add them here
        $topUpWords = 0;
        if (method_exists($this, 'topUps')) {
            $topUpWords = $this->topUps()->where('status', 'active')->sum('words');
        }
        
        return $planWords + $topUpWords;
    }
    public function topUps()
    {
        return $this->hasMany(AiTopUp::class, 'subscription_id');
    }
}