<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class SuperAdmin extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    protected $guard = 'superadmin';

    protected $fillable = [
        'name',
        'email',
        'password',
        'is_active',
        'failed_login_attempts',
        'locked_until',
        'last_login_ip',
        'last_login_at',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'email_verified_at' => 'datetime',
        'locked_until' => 'datetime',
        'last_login_at' => 'datetime',
        'failed_login_attempts' => 'integer',
    ];

    // Relationship with login logs
    public function loginLogs()
    {
        return $this->hasMany(SuperAdminLoginLog::class);
    }

    // Check if account is locked
    public function isLocked()
    {
        return $this->locked_until && $this->locked_until->isFuture();
    }

    // Increment failed attempts
    public function incrementFailedAttempts()
    {
        $this->failed_login_attempts++;
        
        // Lock account after 5 failed attempts for 30 minutes
        if ($this->failed_login_attempts >= 5) {
            $this->locked_until = now()->addMinutes(30);
        }
        
        $this->save();
    }

    // Reset failed attempts on successful login
    public function resetFailedAttempts()
    {
        $this->failed_login_attempts = 0;
        $this->locked_until = null;
        $this->save();
    }
}